<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Menu;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;
use DataTables;
use Illuminate\Database\QueryException;
use Session;

class RoleController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $roles = Role::where('is_deleted','!=',1);

            if ($request->name) {
                $roles->where("name", "like", "%" . $request->name . "%");
            }

            $results = $roles->get();

            return DataTables::of($results)
                ->addIndexColumn()
                ->addColumn("name", function ($data) {
                    if (empty($data->name)) {
                        return "N/A";
                    }
                    return $data->name;
                })
                ->addColumn("action", function ($row) {
                    $btn =
                        '<a class="btn btn-info btn-xs" href="' .
                        route("role.edit", $row->id) .
                        '" data-toggle="tooltip" data-placement="top" title="Edit role"><i class="fa fa-edit"></i> Edit</a>&nbsp;';
                    $btn .=
                        '<a class="btn btn-danger btn-xs deleterole" href="javascript:;" data-role="' .
                        $row->id .
                        '" data-toggle="tooltip" data-placement="top" title="Delete role"><i class="fa fa-trash-o"></i> Delete</a>';

                    return $btn;
                })
                ->rawColumns(["action"])
                ->make(true);
        }

        $data["title"] = "Manage Role";

        return view("pages.admin.role.list", $data);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $permissions = Permission::all()
            ->pluck("name", "id")
            ->toArray();
        $menus = Menu::pluck('menu_name')->toArray();
        return view('pages.admin.role.create',compact("permissions","menus"));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            "role_name" => 'required|regex:/^[\pL\s\-]+$/u|unique:roles,name',
        ]);
        try{
            $role = Role::create([
                "name" => $request->role_name,
                "guard_name" => "web",
                "created_at" => date("Y-m-d H:i:s"),
                "updated_at" => date("Y-m-d H:i:s"),
            ]);
            $permissions = $request->permissions;
            if ($permissions) {
                foreach ($permissions as $permission) {
                    $role->givePermissionTo($permission);
                }
            }
        }
        catch(QueryException $ex) 
        {
            return redirect()
                ->route("role.index")
                ->with("failed", $ex->getMessage());                 
        }   
        Session::flash("success", "Role Added successfully.");

        return redirect()->route("role.index");
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $menus = Menu::pluck('menu_name')->toArray();
        $roleInfo = Role::findOrFail($id); //
        $assignedPermission = [];
        foreach ($roleInfo->permissions as $rolePermission) {
            $assignedPermission[] = $rolePermission->name;
        }

        $permissions = Permission::all()
            ->pluck("name", "id")
            ->toArray();

        return view(
            "pages.admin.role.edit",
            compact("roleInfo", "permissions", "assignedPermission","menus")
        );
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            "name" =>
                'required|regex:/^[\pL\s\-]+$/u|unique:roles,name,' . $id . "",
        ]);
        try{
            $role = Role::find($id);
            $role->name = $request->name;
            $role->guard_name = "web";
            $role->save();
            $p_all = Permission::all();

            foreach ($p_all as $p) {
                $role->revokePermissionTo($p->name);
            }
            $permissions = $request["permissions"];
            if ($permissions) {
                foreach ($permissions as $permission) { echo $permission; echo '======';
                    $role->givePermissionTo($permission);
                }
            }
            
        }
        catch(QueryException $ex) 
        {
            return redirect()
                ->route("role.index")
                ->with("failed", $ex->getMessage());                 
        } 
        Session::flash("success", "Role Updated successfully.");
        return redirect()->route("role.index");
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request)
    {
        $role = Role::find($request->id);
        $role->is_deleted = 1;
        $role->deleted_at = date('Y-m-d H:i:s');
        // $role->delete();
        $role->save();

        Session::flash("success", "Role deleted successfully.");
        return redirect()->route("role.index");
    }
}
