<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Merchant;
use App\Models\ProductCategory;
use Illuminate\Http\Request;
use DataTables;
use Session;
use Illuminate\Support\Facades\Hash;
use App\Models\User;
use Illuminate\Database\QueryException;
use Spatie\Permission\Models\Role;

class UserController extends Controller
{
    public function __construct()
    {
        $this->userStatus = ["1" => "Active", "2" => "Inactive"];
        $this->defaultPassword = "Suhu@123";
        $this->isApproved = ["0" => "Pending", "1" => "Approved","2"=>"Rejected"];
        $this->merchantStatus = ["1" => "Active", "2" => "Inactive"];
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
            
        if ($request->ajax()) {
            
            $results = User::isNotDeleted()->get();

            return DataTables::of($results)
                ->addIndexColumn()
                ->addColumn("user_name", function ($data) {
                    if (empty($data->user_name)) {
                        return "N/A";
                    }
                    return $data->user_name;
                })
               
                ->addColumn("email", function ($data) {
                    if (empty($data->email)) {
                        return "N/A";
                    }
                    return $data->email;
                })
                ->addColumn("role", function ($data) {
                    if (empty($data->role)) {
                        return "N/A";
                    }
                    return $data->role;
                })
                ->addColumn("action", function ($data) {
                    $btn =
                        '<a class="btn btn-info btn-xs" href="' .
                        route("user.edit", $data->id) .
                        '" data-toggle="tooltip" data-placement="top" title="Edit User"><i class="fa fa-edit"></i> Edit</a>&nbsp;';
                    $btn .=
                        '<a class="btn btn-danger btn-xs deleteuser" href="javascript:;" data-user="' .
                        $data->id .
                        '" data-toggle="tooltip" data-placement="top" title="Delete User"><i class="fa fa-trash"></i> Delete</a>';
                    return $btn;
                })
                ->rawColumns(["action"])
                ->make(true);
        }

        $data["title"] = "Manage User";

        return view("pages.admin.user.list", $data);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $data["roles"] = Role::all();
        $data["password"] = $this->defaultPassword;
        return view('pages.admin.user.create',$data);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            "user_name" => "required|regex:/^[a-zA-Z0 ]*$/|max:30",
            "email" => "required|email|unique:users,email",
            "password" => "required",           
            "role" => "required",
        ]);
        $password = Hash::make($request->password);
        try
        {
            $user = User::create([            
                "user_name" => trim($request->user_name),
                "email" => trim($request->email),                
                "password" => $password,
                "role" => $request->role,                
            ]);
            $user->assignRole($request->role);
        }
        catch(QueryException $ex) 
        {
            return redirect()
                ->route("user.index")
                ->with("failed", $ex->getMessage());                 
        }       
        return redirect()
            ->route("user.index")
            ->with("success", "User created successfully!");
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $userInfo = User::where("id", "=", $id)->first();
        $data["userInfo"] = $userInfo;
        $data["roles"] = Role::all();
        return view('pages.admin.user.edit',$data);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            "user_name" => "required|unique:users,user_name,|regex:/^[a-zA-Z0 ]*$/|max:30" . $id . "",
            "email" => "required|email|unique:users,email," . $id . "",
            "role" => "required"
        ]);

        try
        {
            $user = User::find($id);
            $user->user_name = $request->user_name;
            $user->email = $request->email;            
            $user->role = $request->role;
            $user->save();
            $user->syncRoles($request->role);            
        }
        catch(QueryException $ex) 
        {
            return redirect()
                ->route("user.index")
                ->with("failed", "update error");                 
        } 
        
        return redirect()
            ->route("user.index")
            ->with("success", "User updated successfully!");

    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request)
    {
        $user = User::find($request->id);
        $user->delete();
        Session::flash("success", "User information deleted successfully.");
        return 1;
    }
    public function editProfileForm()
    {
        if(\Auth::user()->can('edit profile'))
        {
            if(\Auth::user()->role == "MERCHANT")
            {
                $merchantInfo = Merchant::where('merchant_id', \Auth::user()->merchant->merchant_id)->first();
                if (!$merchantInfo)
                {
                    return redirect()->route('merchant.index');
                }
                $data['merchant_id'] = \Auth::user()->merchant->merchant_id;
                $data['merchantInfo'] = $merchantInfo;

                $data["status"] = $this->merchantStatus;
                $data["password"] = $this->defaultPassword;
                $data["product_category"] = ProductCategory::get();
                return view('pages.admin.merchant.edit-profile',$data);
            }
            else
            {
                return view('pages.admin.user.edit-profile');
            }
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
        
    }
}
